"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const graphql_1 = require("graphql");
const graphql_tag_1 = require("graphql-tag");
const graphql_tools_1 = require("graphql-tools");
const rate_limiter_flexible_1 = require("rate-limiter-flexible");
/**
 * Get a value to uniquely identify a field in a schema.
 * @param directiveArgs The arguments defined in the schema for the directive.
 * @param obj The previous result returned from the resolver on the parent field.
 * @param args The arguments provided to the field in the GraphQL operation.
 * @param context Contains per-request state shared by all resolvers in a particular operation.
 * @param info Holds field-specific information relevant to the current operation as well as the schema details.
 */
function defaultKeyGenerator(directiveArgs, obj, args, context, info) {
    return `${info.parentType.name}.${info.fieldName}`;
}
exports.defaultKeyGenerator = defaultKeyGenerator;
/**
 * Calculate the number of points to consume.
 * @param directiveArgs The arguments defined in the schema for the directive.
 * @param obj The previous result returned from the resolver on the parent field.
 * @param args The arguments provided to the field in the GraphQL operation.
 * @param context Contains per-request state shared by all resolvers in a particular operation.
 * @param info Holds field-specific information relevant to the current operation as well as the schema details.
 */
function defaultPointsCalculator(directiveArgs, obj, args, context, info) {
    return 1;
}
exports.defaultPointsCalculator = defaultPointsCalculator;
/**
 * Raise a rate limit error when there are too many requests.
 * @param resource The current rate limit information for this field.
 * @param directiveArgs The arguments defined in the schema for the directive.
 * @param obj The previous result returned from the resolver on the parent field.
 * @param args The arguments provided to the field in the GraphQL operation.
 * @param context Contains per-request state shared by all resolvers in a particular operation.
 * @param info Holds field-specific information relevant to the current operation as well as the schema details.
 */
function defaultOnLimit(resource, directiveArgs, obj, args, context, info) {
    throw new graphql_1.GraphQLError(`Too many requests, please try again in ${Math.ceil(resource.msBeforeNext / 1000)} seconds.`);
}
exports.defaultOnLimit = defaultOnLimit;
/**
 * Create a GraphQL directive type definition.
 * @param directiveName Name of the directive
 */
function createRateLimitTypeDef(directiveName = 'rateLimit') {
    return graphql_tag_1.default `
  """
  Controls the rate of traffic.
  """
  directive @${directiveName}(
    """
    Number of occurrences allowed over duration.
    """
    limit: Int! = 60

    """
    Number of seconds before limit is reset.
    """
    duration: Int! = 60
  ) on OBJECT | FIELD_DEFINITION
`;
}
exports.createRateLimitTypeDef = createRateLimitTypeDef;
/**
 * Create an implementation of a rate limit directive.
 */
function createRateLimitDirective({ keyGenerator = defaultKeyGenerator, pointsCalculator = defaultPointsCalculator, onLimit = defaultOnLimit, limiterClass = rate_limiter_flexible_1.RateLimiterMemory, limiterOptions = {}, } = {}) {
    const limiters = new Map();
    const limiterKeyGenerator = ({ limit, duration }) => `${limit}/${duration}s`;
    return class extends graphql_tools_1.SchemaDirectiveVisitor {
        // Use createRateLimitTypeDef until graphql-tools fixes getDirectiveDeclaration
        // public static getDirectiveDeclaration(
        //   directiveName: string,
        //   schema: GraphQLSchema,
        // ): GraphQLDirective {
        //   return new GraphQLDirective({
        //     name: directiveName,
        //     description: 'Controls the rate of traffic.',
        //     locations: [
        //       DirectiveLocation.FIELD_DEFINITION,
        //       DirectiveLocation.OBJECT,
        //     ],
        //     args: {
        //       limit: {
        //         type: GraphQLNonNull(GraphQLInt),
        //         defaultValue: 60,
        //         description: 'Number of occurrences allowed over duration.',
        //       },
        //       duration: {
        //         type: GraphQLNonNull(GraphQLInt),
        //         defaultValue: 60,
        //         description: 'Number of seconds before limit is reset.',
        //       },
        //     },
        //   });
        // }
        visitObject(object) {
            // Wrap fields for limiting that don't have their own @rateLimit
            const fields = object.getFields();
            Object.values(fields).forEach(field => {
                if (!field.astNode)
                    return;
                const directives = field.astNode.directives;
                if (!directives ||
                    !directives.some(directive => directive.name.value === this.name)) {
                    this.rateLimit(field);
                }
            });
        }
        visitFieldDefinition(field) {
            this.rateLimit(field);
        }
        getLimiter() {
            const limiterKey = limiterKeyGenerator(this.args);
            let limiter = limiters.get(limiterKey);
            if (limiter === undefined) {
                limiter = new limiterClass(Object.assign(Object.assign({}, limiterOptions), { keyPrefix: limiterOptions.keyPrefix === undefined
                        ? this.name // change the default behaviour which is to use 'rlflx'
                        : limiterOptions.keyPrefix, points: this.args.limit, duration: this.args.duration }));
                limiters.set(limiterKey, limiter);
            }
            return limiter;
        }
        rateLimit(field) {
            const { resolve = graphql_1.defaultFieldResolver } = field;
            const limiter = this.getLimiter();
            field.resolve = (obj, args, context, info) => __awaiter(this, void 0, void 0, function* () {
                const pointsToConsume = yield pointsCalculator(this.args, obj, args, context, info);
                if (pointsToConsume !== 0) {
                    const key = yield keyGenerator(this.args, obj, args, context, info);
                    try {
                        yield limiter.consume(key, pointsToConsume);
                    }
                    catch (e) {
                        if (e instanceof Error) {
                            throw e;
                        }
                        const resource = e;
                        return onLimit(resource, this.args, obj, args, context, info);
                    }
                }
                return resolve.apply(this, [obj, args, context, info]);
            });
        }
    };
}
exports.createRateLimitDirective = createRateLimitDirective;
//# sourceMappingURL=index.js.map