"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
function __export(m) {
    for (var p in m) if (!exports.hasOwnProperty(p)) exports[p] = m[p];
}
Object.defineProperty(exports, "__esModule", { value: true });
const parsers_1 = require("../parsers");
const promise_or_callback_1 = require("../promise-or-callback");
const search_resource_1 = require("../search-resource");
const builders_1 = require("./builders");
const index_buffer_1 = require("./index-buffer");
const index_stream_1 = require("./index-stream");
__export(require("azure-search-types/dist/indexes/search"));
const RE_DATE = /^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}(?:\.\d{1,3})?Z$/;
class SearchIndex extends search_resource_1.SearchResource {
    /**
     * Manage Azure Search index resources
     * @param requester http handler
     * @param type must be 'indexes'
     * @param name the name of the current search index
     */
    constructor(requester, type, name) {
        super(requester, type, name);
    }
    buildQuery() {
        return new builders_1.QueryBuilder(this);
    }
    search(query, optionsOrCallback, callback) {
        const options = typeof optionsOrCallback === 'function' ? {} : optionsOrCallback || {};
        if (options.parseDates === undefined) {
            options.parseDates = true;
        }
        return this.request({
            method: 'post',
            path: '/docs/search',
            body: query,
            parser: parsers_1.jsonParser((key, value) => {
                if (options.parseDates && typeof value === 'string' && RE_DATE.test(value)) {
                    value = new Date(value);
                }
                return value;
            }),
        }, optionsOrCallback, callback);
    }
    suggest(query, optionsOrCallback, callback) {
        const options = typeof optionsOrCallback === 'function' ? {} : optionsOrCallback || {};
        if (options.parseDates === undefined) {
            options.parseDates = true;
        }
        return this.request({
            method: 'post',
            path: '/docs/suggest',
            body: query,
            parser: parsers_1.jsonParser((key, value) => {
                if (options.parseDates && typeof value === 'string' && RE_DATE.test(value)) {
                    value = new Date(value);
                }
                return value;
            }),
        }, optionsOrCallback, callback);
    }
    analyze(query, optionsOrCallback, callback) {
        return this.request({
            method: 'post',
            path: '/analyze',
            body: query,
        }, optionsOrCallback, callback);
    }
    index(documents, optionsOrCallback1, optionsOrCallback2, callback) {
        return __awaiter(this, void 0, void 0, function* () {
            let options;
            let batchIndexed;
            let cb;
            let count = 0;
            // full callback overload
            if (typeof callback === 'function') {
                options = optionsOrCallback2;
                batchIndexed = null;
                cb = callback;
                // callback overload without options
            }
            else if (typeof optionsOrCallback2 === 'function') {
                options = {};
                batchIndexed = optionsOrCallback1;
                cb = optionsOrCallback2;
                // callback overload without options or batchIndexed
            }
            else if (typeof optionsOrCallback1 === 'function') {
                options = {};
                batchIndexed = null;
                cb = optionsOrCallback1;
                // promise overload
            }
            else {
                options = optionsOrCallback2;
                batchIndexed = optionsOrCallback1;
                cb = null;
            }
            return yield promise_or_callback_1.promiseOrCallback(() => __awaiter(this, void 0, void 0, function* () {
                const buffer = new index_buffer_1.IndexBuffer((data) => __awaiter(this, void 0, void 0, function* () {
                    const resp = yield this.request({
                        method: 'post',
                        path: '/docs/index',
                        headers: { 'content-type': 'application/json' },
                        body: data,
                    }, options);
                    if (batchIndexed) {
                        yield batchIndexed(resp.result.value);
                    }
                    count += resp.result.value.length;
                    return resp.result.value;
                }));
                for (const document of documents) {
                    yield buffer.add(document);
                }
                yield buffer.flush();
                return count;
            }), cb);
        });
    }
    /**
     * Create a indexing stream suitable for piping in document objects.
     * Emits a 'data' event for every batch of documents that is indexed.
     * @param batchIndexed optional callback when a batch has been indexed
     * @param options optional request options
     */
    createIndexingStream(options) {
        return new index_stream_1.IndexStream(this.requester, this.name, options);
    }
    statistics(optionsOrCallback, callback) {
        return this.request({
            method: 'get',
            path: '/stats',
        }, optionsOrCallback, callback);
    }
    lookup(key, optionsOrCallback, callback) {
        const options = typeof optionsOrCallback === 'function' ? {} : optionsOrCallback || {};
        if (options.parseDates === undefined) {
            options.parseDates = true;
        }
        return this.request({
            method: 'get',
            path: `/docs/${key}`,
            parser: parsers_1.jsonParser((key, value) => {
                if (options.parseDates && typeof value === 'string' && RE_DATE.test(value)) {
                    value = new Date(value);
                }
                return value;
            }),
        }, optionsOrCallback, callback);
    }
    count(optionsOrCallback, callback) {
        return this.request({
            method: 'get',
            path: '/docs/$count',
            headers: { accept: 'text/plain' },
            parser: parsers_1.jsonParser(),
        }, optionsOrCallback, callback);
    }
    facet(field) {
        return new builders_1.QueryFacet(field);
    }
}
exports.SearchIndex = SearchIndex;
//# sourceMappingURL=search-index.js.map